<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use App\Services\QuickBooksService;

class PaymentController extends Controller
{
    public function createPaymentIntent(Request $request)
    {
        $request->validate([
            'amount' => 'required|numeric|min:0.50',
            'currency' => 'required|string|size:3',
        ]);

        $stripeSecretKey = env('STRIPE_SECRET');

        $response = Http::withToken($stripeSecretKey)->post('https://api.stripe.com/v1/payment_intents', [
            'amount' => $request->amount * 100, // Convert to cents
            'currency' => $request->currency,
            'payment_method_types' => ['card'],
        ]);

        if ($response->failed()) {
            return response()->json(['error' => 'Failed to create payment intent.'], 500);
        }

        return response()->json($response->json());
    }

    public function processPayment(Request $request)
    {
        $request->validate([
            'card_number' => 'required|string',
            'expiry_date' => 'required|string',
            'cvc' => 'required|string',
            'amount' => 'required|numeric|min:0.50',
            'currency' => 'required|string|size:3',
        ]);

        $stripeSecretKey = env('STRIPE_SECRET');

        $expiry = explode('/', $request->expiry_date);
        $expiryMonth = trim($expiry[0]);
        $expiryYear = trim($expiry[1]);

        $response = Http::withToken($stripeSecretKey)->post('https://api.stripe.com/v1/payment_intents', [
            'amount' => $request->amount * 100, // Convert to cents
            'currency' => $request->currency,
            'payment_method_data' => [
                'type' => 'card',
                'card' => [
                    'number' => $request->card_number,
                    'exp_month' => $expiryMonth,
                    'exp_year' => $expiryYear,
                    'cvc' => $request->cvc,
                ],
            ],
            'confirm' => true,
        ]);

        if ($response->failed()) {
            return response()->json(['error' => 'Payment failed.'], 500);
        }

        return response()->json(['success' => 'Payment processed successfully.', 'data' => $response->json()]);
    }

    public function handleStripeWebhook(Request $request)
    {
        $stripeSecretKey = env('STRIPE_SECRET');
        $endpointSecret = env('STRIPE_WEBHOOK_SECRET');

        $signature = $request->header('Stripe-Signature');
        $payload = $request->getContent();

        try {
            $event = \Stripe\Webhook::constructEvent(
                $payload,
                $signature,
                $endpointSecret
            );
        } catch (\UnexpectedValueException $e) {
            return response()->json(['error' => 'Invalid payload'], 400);
        } catch (\Stripe\Exception\SignatureVerificationException $e) {
            return response()->json(['error' => 'Invalid signature'], 400);
        }

        if ($event->type === 'payment_intent.succeeded') {
            $paymentIntent = $event->data->object;
            // Handle successful payment intent (e.g., update order status)
        }

        return response()->json(['status' => 'success']);
    }
}